<?php

namespace App\Http\Controllers;

use App\Models\Rate;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class RateController extends Controller
{
    public function index(): JsonResponse
    {
        $rates = Rate::with(['candidateProfile', 'currency'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $rates,
        ], 200);
    }

    public function show($profileId): JsonResponse
    {
        $rate = Rate::where('profile_id', $profileId)->first();

        if (!$rate) {
            return response()->json([
                'status' => 'error',
                'message' => 'Rate not found for this profile',
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => $rate,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        // Disable store method, use PUT instead
        return response()->json([
            'status' => 'error',
            'message' => 'Use PUT request to /api/rates/{profileId} for create/update operations.',
        ], 405); // Method Not Allowed
    }

    public function update(Request $request, $profileId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'hourly_min' => [
                'sometimes',
                'numeric',
                'min:0',
                function ($attribute, $value, $fail) use ($request) {
                    if ($request->input('is_negotiable') !== true && !$request->has('hourly_min')) {
                        $fail('The hourly min is required when is_negotiable is not true.');
                    }
                },
            ],
            'hourly_max' => [
                'sometimes',
                'numeric',
                'min:0',
                function ($attribute, $value, $fail) use ($request) {
                    if ($request->input('is_negotiable') !== true) {
                        if (!$request->has('hourly_max')) {
                            $fail('The hourly max is required when is_negotiable is not true.');
                        } elseif ($value <= $request->input('hourly_min')) {
                            $fail('The hourly max must be greater than hourly min.');
                        }
                    }
                },
            ],
            'currency_id' => 'sometimes|exists:currencies,Currency_id',
            'is_negotiable' => 'sometimes|boolean',
            'factors' => 'sometimes|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Fetch rate for the profile
        $rate = Rate::where('profile_id', $profileId)->first();

        $data = $request->only([
            'hourly_min',
            'hourly_max',
            'currency_id',
            'is_negotiable',
            'factors',
        ]);

        if ($rate) {
            // If record exists and new data is provided, delete and recreate
            if (!empty(array_filter($data))) {
                $rate->delete();
                $responseData = Rate::create(array_merge($data, ['profile_id' => $profileId]));
            } else {
                // If no new data, return existing
                $responseData = $rate;
            }
        } else {
            // If no record exists, create a new one
            $responseData = Rate::create(array_merge($data, ['profile_id' => $profileId]));
        }

        return response()->json([
            'status' => 'success',
            'data' => $responseData,
        ], $rate && !empty(array_filter($data)) ? 201 : 200); // 201 for replace, 200 for no change or initial create
    }
    public function destroy($profileId): JsonResponse
    {
        $rate = Rate::where('profile_id', $profileId)->first();

        if (!$rate) {
            return response()->json([
                'status' => 'error',
                'message' => 'Rate not found for this profile',
            ], 404);
        }

        $rate->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Rate deleted successfully',
        ], 200);
    }
}
